<?php

namespace App\Http\Controllers\Api;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Payment;
use App\Models\Product;
use App\Models\MenuItem;
use App\Models\StockMove;
use App\Models\Discount;
use App\Models\Refund;
use App\Models\RefundItem;

class OrderController extends BaseApiController
{
    private function resolveDiscount(?array $d, int $subtotal): array
    {
        if (!$d) return [0, null];
        $payload = $d;
        if (!empty($d['code'])) {
            $found = Discount::where('code',$d['code'])->first();
            if ($found && $found->enabled) {
                $payload = ['code'=>$found->code,'kind'=>$found->kind,'value'=>$found->value];
                if ($found->kind === 'percent') {
                    $off = (int) round($subtotal * (float)$found->value / 100.0);
                    return [min($subtotal, $off), $payload];
                } else {
                    return [min($subtotal, (int)round($found->value)), $payload];
                }
            }
        }
        if (!empty($d['kind']) && isset($d['value'])) {
            if ($d['kind'] === 'percent') {
                $off = (int) round($subtotal * (float)$d['value'] / 100.0);
                return [min($subtotal, $off), $payload];
            } else {
                return [min($subtotal, (int)round($d['value'])), $payload];
            }
        }
        return [0, null];
    }

    public function store(Request $r)
    {
        $data = $r->validate([
            'items' => 'required|array|min:1',
            'items.*.qty' => 'required|integer|min:1',
            'items.*.price_cents' => 'required|integer|min:0',
            'items.*.sku' => 'nullable|string',
            'items.*.menu_code' => 'nullable|string',
            'items.*.name' => 'nullable|string',
            'items.*.temp' => 'nullable|in:ice,hot',
            'items.*.size' => 'nullable|in:S,M,L',
            'tax_rate' => 'nullable|numeric',
            'service_rate'=> 'nullable|numeric',
            'payments' => 'array',
            'payments.*.method' => 'required|string|max:50',
            'payments.*.amount_cents' => 'required|integer|min:0',
            'discount' => 'nullable|array',
            'order_type' => 'nullable|string|max:50',
        ]);

        $uId = $this->currentUser($r)?->id;

        $subtotal = collect($data['items'])->sum(fn($it) => $it['price_cents'] * $it['qty']);
        [$discountCents, $discountMeta] = $this->resolveDiscount($data['discount'] ?? null, $subtotal);

        // $taxRate = $data['tax_rate'] ?? 0;
        // $tax = (int) round(($subtotal - $discountCents) * $taxRate);
        // $total = ($subtotal - $discountCents) + $tax;


        $netSubtotal = $subtotal - $discountCents;

        $taxRate = (float) ($data['tax_rate'] ?? 0);
        $serviceRate = (float) ($data['service_rate']?? 0);
        $serviceCents = (int) round($netSubtotal * $serviceRate);
        $totalService = $netSubtotal + $serviceCents;

        $tax = (int) round($totalService * $taxRate);

        $total = $totalService + $tax;


        $orderType = $data['order_type'] ?? ($data['sales_type'] ?? null) ?? ($data['type'] ?? null);

        $order = DB::transaction(function () use ($data, $uId, $subtotal, $discountCents,$serviceCents, $tax, $total, $orderType, $discountMeta) {
            // === simpan order utama
            $order = Order::create([
                'subtotal_cents' => $subtotal,
                'discount_cents' => $discountCents,
                'service_cents' => $serviceCents,
                'tax_cents'      => $tax,
                'total_cents'    => $total,
                'payload'        => json_encode($data),
                'created_by'     => $uId,
                'order_type'     => $orderType,
            ]);

            // === simpan item + pengurangan stok
            foreach ($data['items'] as $it) {
                OrderItem::create([
                    'order_id'    => $order->id,
                    'sku'         => $it['sku'] ?? null,
                    'menu_code'   => $it['menu_code'] ?? null,
                    'qty'         => $it['qty'],
                    'price_cents' => $it['price_cents'],
                    'name'        => $it['name'] ?? null,
                    'temp'        => $it['temp'] ?? null,
                    'size'        => $it['size'] ?? null,
                    'created_by'  => $uId,
                ]);

                // === logika pengurangan stok
                if (!empty($it['sku'])) {
                    $p = Product::where('sku', $it['sku'])->first();
                    if ($p) {
                        $reduce = $it['qty'];
                        $p->decrement('stock', $reduce);
                        StockMove::create([
                            'sku'        => $p->sku,
                            'delta'      => -$reduce,
                            'reason'     => 'order direct item',
                            'created_by' => $uId,
                        ]);
                    }
                } elseif (!empty($it['menu_code'])) {
                    $components = MenuItem::where('menu_code', $it['menu_code'])->get(['product_sku', 'qty']);
                    foreach ($components as $c) {
                        $reduceBy = $c->qty * $it['qty'];
                        $p = Product::where('sku', $c->product_sku)->first();
                        if ($p) {
                            $p->decrement('stock', $reduceBy);
                            StockMove::create([
                                'sku'        => $p->sku,
                                'delta'      => -$reduceBy,
                                'reason'     => 'order menu component',
                                'created_by' => $uId,
                            ]);
                        }
                    }
                }
            }

            // === simpan pembayaran (jika ada)
            if (!empty($data['payments'])) {
                foreach ($data['payments'] as $p) {
                    Payment::create([
                        'order_id'     => $order->id,
                        'method'       => $p['method'],
                        'amount_cents' => (int) $p['amount_cents'],
                        'created_by'   => $uId,
                    ]);
                }
            }

            return $order;
        });

        return response()->json([
            'ok'          => true,
            'id'          => $order->id,
            'total_cents' => $order->total_cents,
        ]);
    }

    public function index(Request $r)
    {
        $u = $this->currentUser($r);
        $since = $r->query('since');

        $q = Order::query();

        if ($u) {
            $q->where('created_by', $u->id);
        }

        if (!empty($since)) {
            $q->where('created_at', '>=', $since);
        }

        // ===== NEW: penanda order yang sudah direfund + total nominal refund
        $q->addSelect([
            'refunds_count' => Refund::selectRaw('COUNT(*)')
                ->whereColumn('order_id', 'orders.id'),
            'refund_total_cents' => Refund::selectRaw('COALESCE(SUM(total_cents),0)')
                ->whereColumn('order_id', 'orders.id'),
        ]);

        return $q->orderByDesc('id')->get([
            'id',
            'subtotal_cents',
            'discount_cents',
            'service_cents',
            'tax_cents',
            'total_cents',
            'order_type',
            'payload',
            'created_at',
            // kolom subselect juga ikut terambil:
            'refunds_count',
            'refund_total_cents',
        ]);
    }

    public function show(Request $r, $id)
    {
        $u = $this->currentUser($r);
        $o = Order::findOrFail($id);

        if ($u && $o->created_by && $o->created_by !== $u->id) {
            return response()->json(['ok'=>false,'message'=>'forbidden'], 403);
        }

        $items = $o->items()->get();
        $payments = $o->payments()->get();

        // ===== NEW: agregat qty yang sudah direfund per order_item
        $refundedAgg = RefundItem::where('order_id', $o->id)
            ->groupBy('order_item_id')
            ->select('order_item_id', DB::raw('COALESCE(SUM(qty),0) as refunded_qty'))
            ->pluck('refunded_qty', 'order_item_id'); // [order_item_id => qty]

        // ===== NEW: sematkan penanda di setiap item
        $items = $items->map(function ($it) use ($refundedAgg) {
            $rid = (int)($refundedAgg[$it->id] ?? 0);
            $it->setAttribute('refunded_qty', $rid);
            $it->setAttribute('refunded', $rid > 0);
            return $it;
        });

        // (opsional) total refund & ada refund?
        $refundsCount = Refund::where('order_id',$o->id)->count();
        $refundTotal = (int) Refund::where('order_id',$o->id)->sum('total_cents');

        return [
            'ok' => true,
            'order' => $o,
            'items' => $items,
            'payments' => $payments,
            'has_refund' => $refundsCount > 0,          // NEW
            'refunds_count' => $refundsCount,            // NEW
            'refund_total_cents' => $refundTotal,        // NEW
        ];
    }

    // ===== daftar refund + itemnya (tetap) =====
    public function refunds(Request $r, $id)
    {
        $order = Order::with('refunds.items')->findOrFail($id);
        return [
            'ok'=>true,
            'refunds'=>$order->refunds()->with('items')->orderByDesc('id')->get()
        ];
    }

    // ===== Buat refund parsial/total =====
    public function refund(Request $r, $id)
    {
        $data = $r->validate([
            'items' => 'required|array|min:1',
            'items.*.order_item_id'    => 'nullable|integer',
            'items.*.sku'              => 'nullable|string',
            'items.*.menu_code'        => 'nullable|string',
            'items.*.qty'              => 'required|integer|min:1',
            'items.*.unit_price_cents' => 'nullable|integer|min:0',
            'reason' => 'nullable|string|max:200',
        ]);

        $order = Order::with('items')->findOrFail($id);
        $orderItems = $order->items->keyBy('id');

        // total qty yang sudah direfund per order_item
        $refundedSoFar = RefundItem::where('order_id',$order->id)
            ->selectRaw('order_item_id, COALESCE(SUM(qty),0) as qty')
            ->groupBy('order_item_id')
            ->pluck('qty','order_item_id');

        // Validasi qty ≤ (dibeli - sudah_refund)
        foreach ($data['items'] as $it) {
            $oiId = $it['order_item_id'] ?? null;
            if ($oiId && isset($orderItems[$oiId])) {
                $bought  = (int)$orderItems[$oiId]->qty;
                $already = (int)($refundedSoFar[$oiId] ?? 0);
                $req     = (int)$it['qty'];
                if ($req > max(0, $bought - $already)) {
                    return response()->json(['ok'=>false,'message'=>"refund qty exceeds purchased for item $oiId"], 422);
                }
            }
        }

        // // Hitung total refund (fallback ambil harga dari order_item)
        // $total = 0;
        // foreach ($data['items'] as &$it) {
        //     $unit = (int)($it['unit_price_cents'] ?? 0);
        //     if (!$unit && !empty($it['order_item_id']) && isset($orderItems[$it['order_item_id']])){
        //         $unit = (int)($orderItems[$it['order_item_id']]->price_cents ?? $orderItems[$it['order_item_id']]->price ?? 0);
        //     }
        //     $it['unit_price_cents'] = $unit;
        //     $total += $unit * (int)$it['qty'];
        // }
        // unset($it);

        // $refund = DB::transaction(function () use ($data, $order, $total, $orderItems) {

        //     $refund = Refund::create([
        //         'order_id'    => $order->id,
        //         'total_cents' => $total,
        //         'reason'      => $data['reason'] ?? null,
        //         'payload'     => $data,
        //     ]);

                // Hitung subtotal refund (fallback ambil harga dari order_item)
        $subtotalRefund = 0;
        foreach ($data['items'] as &$it) {
            $unit = (int)($it['unit_price_cents'] ?? 0);

            if (
                !$unit &&
                !empty($it['order_item_id']) &&
                isset($orderItems[$it['order_item_id']])
            ) {
                $oi   = $orderItems[$it['order_item_id']];
                $unit = (int)($oi->price_cents ?? $oi->price ?? 0);
            }

            $it['unit_price_cents'] = $unit;
            $subtotalRefund += $unit * (int)$it['qty'];
        }
        unset($it);

        // ===== Distribusi diskon & pajak proporsional dari order asli =====
        $orderSubtotal = (int)($order->subtotal_cents ?? 0);
        $orderDiscount = (int)($order->discount_cents ?? 0);
        $orderTax      = (int)($order->tax_cents ?? 0);

        $ratio = 0;
        if ($orderSubtotal > 0 && $subtotalRefund > 0) {
            // batasi max 1 biar ga lebih gede dari order asli
            $ratio = min(1, $subtotalRefund / $orderSubtotal);
        }

        $discountRefund = (int) round($orderDiscount * $ratio);
        $taxRefund      = (int) round($orderTax * $ratio);

        // total refund = subtotal - bagian diskon + bagian pajak
        $totalRefund = $subtotalRefund - $discountRefund + $taxRefund;

        // simpan perhitungan di payload buat debugging
        $calc = [
            'subtotal_refund_cents' => $subtotalRefund,
            'discount_refund_cents' => $discountRefund,
            'tax_refund_cents'      => $taxRefund,
            'ratio'                 => $ratio,
        ];

        $refund = DB::transaction(function () use ($data, $order, $totalRefund, $orderItems, $calc) {

            $payload = $data;
            $payload['_calc'] = $calc;

            $refund = Refund::create([
                'order_id'    => $order->id,
                'total_cents' => $totalRefund,
                'reason'      => $data['reason'] ?? null,
                'payload'     => $payload,
            ]);


            // helper: tambah stok + log
            $restockSku = function (string $sku, int $inc) {
                $p = Product::where('sku', $sku)->first();
                if ($p && $inc > 0) {
                    $p->increment('stock', $inc);
                    StockMove::create([
                        'sku'    => $p->sku,
                        'delta'  => +$inc,
                        'reason' => 'refund',
                    ]);
                }
            };

            foreach ($data['items'] as $it) {
                RefundItem::create([
                    'refund_id'        => $refund->id,
                    'order_id'         => $order->id,
                    'order_item_id'    => $it['order_item_id'] ?? null,
                    'sku'              => $it['sku'] ?? null,
                    'menu_code'        => $it['menu_code'] ?? null,
                    'qty'              => (int)$it['qty'],
                    'unit_price_cents' => (int)$it['unit_price_cents'],
                ]);

                // === RESTOCK ===
                if (!empty($it['sku'])) {
                    $restockSku($it['sku'], (int)$it['qty']);
                } elseif (!empty($it['menu_code'])) {
                    $components = MenuItem::where('menu_code', $it['menu_code'])->get(['product_sku','qty']);
                    foreach ($components as $c) {
                        $restockSku($c->product_sku, (int)$c->qty * (int)$it['qty']);
                    }
                } else {
                    // fallback: lewat order_item
                    $oi = !empty($it['order_item_id']) ? ($orderItems[$it['order_item_id']] ?? null) : null;
                    if ($oi && !empty($oi->sku)) {
                        $restockSku($oi->sku, (int)$it['qty']);
                    } elseif ($oi && !empty($oi->menu_code)) {
                        $components = MenuItem::where('menu_code', $oi->menu_code)->get(['product_sku','qty']);
                        foreach ($components as $c) {
                            $restockSku($c->product_sku, (int)$c->qty * (int)$it['qty']);
                        }
                    }
                }
            }

            return $refund;
        });

        return ['ok'=>true, 'refund_id'=>$refund->id, 'total_cents'=>$refund->total_cents];
    }
}
